﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Windows;
using System.Windows.Media;

namespace AnalyseToolPluginContracts
{
    public interface IAnalyseToolPlugin
    {
        /// <summary>
        /// Plugin display name
        /// </summary>
        string Name { get; }
        /// <summary>
        /// Plugin short description
        /// </summary>
        string Description { get; }
        /// <summary>Plugin Icon URI
        /// <example>
        /// <code>
        /// IconUri => new Uri("pack://application:,,,/PluginName;component/Images/icon.png")
        /// </code>
        /// </example>
        /// </summary>
        Uri IconUri { get; }
        /// <summary> Plugin contents <c>IAnalyseToolPluginContent</c>
        /// <list type="bullet">
        /// <item><description>PageData</description></item>
        /// <item><description>TitleBarData</description></item>
        /// <item><description>PopUpData</description></item>
        /// <item><description>OverlayData</description></item>
        /// <item><description>HintData</description></item>
        /// </list>
        /// </summary>
        IEnumerable<IAnalyseToolPluginContent> Contents { get; }
        /// <summary>
        /// Plugin Author
        /// </summary>
        string Author { get; }
        /// <summary>
        /// Plugin Author Contact info
        /// </summary>
        string Contact { get; }
        /// <summary>
        /// Program versions that the plugin is compatible with
        /// <list type="bullet">
        /// <item><description>Between two version<code>"1.0.0/1.1.0"</code></description></item>
        /// <item><description>Fixed versions<code>"1.0.0;1.1.0"</code></description></item>
        /// <item><description>Or combined<code>"0.9.0;1.0.0/1.1.0;1.2.0" </code></description></item>
        /// </list>
        /// </summary>
        string VersionApply { get; }
        /// <summary> Plugin options <c>IAnalyseToolPluginOption</c>
        /// <list type="bullet">
        /// <item><description>BoolPluginOption [TYPE : <c>Boolean</c>]</description></item>
        /// <item><description>NullBoolPluginOption [TYPE : <c>Nullable Boolean</c>]</description></item>
        /// <item><description>ComboBoxPluginOption [TYPE : <c>ComboBox List</c>]</description></item>
        /// <item><description>ColorPluginOption [TYPE : <c>Color</c>]</description></item>
        /// <item><description>StringPluginOption [TYPE : <c>String</c>]</description></item>
        /// <item><description>PathPluginOption [TYPE : <c>Path:String</c>]</description></item>
        /// <item><description>DirectoryPluginOption [TYPE : <c>Directory:String</c>]</description></item>
        /// </list>
        /// </summary>
        IEnumerable<IAnalyseToolPluginOption> Options { get; }
        
    }

    public interface IAnalyseToolPluginContent
    {
        string Name { get; }
    }

    public class PageData : IAnalyseToolPluginContent
    {
        public string Name { get; set; }
        public event PropertyChangedEventHandler PropertyChanged;
        public event EventHandler MainThemeChanged;
        public event EventHandler<object> ShowPage;
        public event EventHandler HidePage;
        private DependencyObject _pageContent;
        public void InvokeThemeChanged() { MainThemeChanged?.Invoke(this, EventArgs.Empty); }
        public void InvokeShowPage() { ShowPage?.Invoke(this, EventArgs.Empty); }
        public void InvokeHidePage() { HidePage?.Invoke(this, EventArgs.Empty); }
        public DependencyObject PageContent 
        { 
            get => _pageContent; 
            set { _pageContent = value; PropertyChanged?.Invoke(this, new PropertyChangedEventArgs(nameof(PageContent))); } 
        }
        
        private bool _isExtendedPage = false;

        public bool IsExtendedPage
        {
            get => _isExtendedPage;
            set { _isExtendedPage = value; PropertyChanged?.Invoke(this, new PropertyChangedEventArgs(nameof(IsExtendedPage))); }
        }
        
        private bool _isOverlay = false;
        public bool IsOverlay
        {
            get => _isOverlay;
            set { _isOverlay = value; PropertyChanged?.Invoke(this, new PropertyChangedEventArgs(nameof(IsOverlay))); }
        }

        private MenuItemData _menuItem;
        public MenuItemData MenuItem
        {
            get => _menuItem;
            set { _menuItem = value; PropertyChanged?.Invoke(this, new PropertyChangedEventArgs(nameof(MenuItem))); }
        }
    }

    public class MenuItemData
    {
        public event EventHandler ClickEvent;
        public event EventHandler SelectedEvent;
        public event EventHandler UnselectedEvent;
        public event EventHandler MainThemeChanged;
        public event PropertyChangedEventHandler PropertyChanged;
        public int MenuItemIndex { get; }
        public MenuItemData(int menuItemIndex) { MenuItemIndex = menuItemIndex; }
        public void InvokeClickEvent() { ClickEvent?.Invoke(this, EventArgs.Empty); }
        public void InvokeThemeChanged() { MainThemeChanged?.Invoke(this, EventArgs.Empty); }
        
        private Visibility _chipVisibility = Visibility.Collapsed;
        public Visibility ChipVisibility
        {
            get => _chipVisibility;
            set { _chipVisibility = value; PropertyChanged?.Invoke(this, new PropertyChangedEventArgs(nameof(ChipVisibility))); }
        }
        
        private bool _isWorkingOn = false;
        public bool IsWorkingOn
        {
            get => _isWorkingOn;
            set { _isWorkingOn = value; PropertyChanged?.Invoke(this, new PropertyChangedEventArgs(nameof(IsWorkingOn))); }
        }
        
        private bool _isSelected = false;
        public bool IsSelected
        {
            get => _isSelected;
            set
            {
                _isSelected = value; 
                PropertyChanged?.Invoke(this, new PropertyChangedEventArgs(nameof(IsSelected)));
                if (value) { SelectedEvent?.Invoke(this, EventArgs.Empty); }
                else { UnselectedEvent?.Invoke(this, EventArgs.Empty); }
            }
        }

        private string _menuItemText = string.Empty;
        public string MenuItemText
        {
            get => _menuItemText;
            set { _menuItemText = value; PropertyChanged?.Invoke(this, new PropertyChangedEventArgs(nameof(MenuItemText))); }
        }
        
        private string _menuItemToolTip = string.Empty;
        public string MenuItemToolTip
        {
            get => _menuItemToolTip;
            set { _menuItemToolTip = value; PropertyChanged?.Invoke(this, new PropertyChangedEventArgs(nameof(MenuItemToolTip))); }
        }

        private bool _isFixedStyle = false;
        public bool IsFixedStyle
        {
            get => _isFixedStyle;
            set { _isFixedStyle = value; PropertyChanged?.Invoke(this, new PropertyChangedEventArgs(nameof(IsFixedStyle))); }
        }
    }

    public interface IAnalyseToolPluginOption
    {
        string Name { get; }
        string VisualName { get; }
        string Description { get; }
        event PropertyChangedEventHandler PropertyChanged;
        bool IsVisible { get; }
        object DefaultValue { get; }
    }
    public class BoolPluginOption : IAnalyseToolPluginOption
    {
        public string Name { get; set; }
        public string VisualName { get; set; }
        public string Description { get; set; }
        public bool IsVisible { get; set; } = true;
        public event PropertyChangedEventHandler PropertyChanged;
        
        private bool _value = false;
        public bool Value
        {
            get => _value; 
            set { _value = value; PropertyChanged?.Invoke(this, new PropertyChangedEventArgs(nameof(Value))); } 
        }
        public object DefaultValue { get; set; } = false;
    }
    public class NullBoolPluginOption : IAnalyseToolPluginOption
    {
        public string Name { get; set; }
        public string VisualName { get; set; }
        public string Description { get; set; }
        public bool IsVisible { get; set; } = true;
        public object DefaultValue { get; set; } = null;
        public event PropertyChangedEventHandler PropertyChanged;
        private bool? _value = null;

        public bool? Value
        {
            get => _value;
            set { _value = value; PropertyChanged?.Invoke(this, new PropertyChangedEventArgs(nameof(Value))); }
        }
    }
    public class ComboBoxPluginOption : IAnalyseToolPluginOption
    {
        public string Name { get; set; }
        public string VisualName { get; set; }
        public string Description { get; set; }
        public bool IsVisible { get; set; } = true;
        public object DefaultValue { get; set; } = null;
        public event PropertyChangedEventHandler PropertyChanged;
        
        private IEnumerable<object> _items;
        public IEnumerable<object> Items
        {
            get => _items;
            set { _items = value; PropertyChanged?.Invoke(this, new PropertyChangedEventArgs(nameof(Items))); }
        }
        
        private object _selectedItem;

        public object SelectedItem
        {
            get => _selectedItem;
            set { _selectedItem = value; PropertyChanged?.Invoke(this, new PropertyChangedEventArgs(nameof(SelectedItem))); }
        }
    }
    public class ColorPluginOption : IAnalyseToolPluginOption
    {
        public string Name { get; set; }
        public string VisualName { get; set; }
        public string Description { get; set; }
        public bool IsVisible { get; set; } = true;
        public object DefaultValue { get; set; } = null;
        public event PropertyChangedEventHandler PropertyChanged;
        private Color _color;

        public Color Color
        {
            get => _color;
            set { _color = value; PropertyChanged?.Invoke(this, new PropertyChangedEventArgs(nameof(Color))); }
        }

        public void SetColor(Color color) { Color = color; }
        public void SetColor(SolidColorBrush brush) { Color = brush.Color; }
    }
    public class StringPluginOption : IAnalyseToolPluginOption
    {
        public string Name { get; set; }
        public string VisualName { get; set; }
        public string Description { get; set; }
        public bool IsVisible { get; set; } = true;
        public object DefaultValue { get; set; } = null;
        public event PropertyChangedEventHandler PropertyChanged;
        private string _text;

        public string Text
        {
            get => _text;
            set { _text = value; PropertyChanged?.Invoke(this, new PropertyChangedEventArgs(nameof(Text))); }
        }
    }
    public class PathPluginOption : IAnalyseToolPluginOption
    {
        public string Name { get; set; }
        public string VisualName { get; set; }
        public string Description { get; set; }
        public bool IsVisible { get; set; } = true;
        public object DefaultValue { get; set; } = null;
        public event PropertyChangedEventHandler PropertyChanged;
        private string _path;

        public string Path
        {
            get => _path;
            set { _path = value; PropertyChanged?.Invoke(this, new PropertyChangedEventArgs(nameof(Path))); }
        }
        public string FileFormats { get; set; }
        public string InitialDirectory { get; set; }
        public string DialogTitle { get; set; }
    }
    public class DirectoryPluginOption : IAnalyseToolPluginOption
    {
        public string Name { get; set; }
        public string VisualName { get; set; }
        public string Description { get; set; }
        public bool IsVisible { get; set; } = true;
        public object DefaultValue { get; set; } = null;
        public event PropertyChangedEventHandler PropertyChanged;
        private string _directory;

        public string Directory
        {
            get => _directory;
            set { _directory = value; PropertyChanged?.Invoke(this, new PropertyChangedEventArgs(nameof(Directory))); }
        }
        public string InitialDirectory { get; set; }
        public string DialogTitle { get; set; }
    }
}