using System;
using System.Collections.Generic;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Media;
using System.Windows.Threading;
using AnalyseToolExtension.DataObjects;
using MaterialDesignThemes.Wpf;

namespace AnalyseToolWpf.DataObjects
{
    public class UnitInfoViewModel
    {
        public string UnitId { get; set; }
        public string NextId { get; set; }
        public string PrevId { get; set; }
        public string IdType { get; set; }
        public string Material { get; set; }
        public string Description { get; set; }
        public string CarrierId { get; set; }
        public string Order { get; set; }
        public bool OrderLock { get; set; }
        public string WipMama { get; set; }
        public Brush WipMamaBrush { get; set; } = Brushes.Transparent;
        public string LockString { get; set; }
        public string WipStation { get; set; }
        public string WipDate {get; set;}
        public Brush WipBoxBrush { get; set; } = Brushes.Transparent;
        public string WipPrefixText { get; set; }
        public string EvaStation { get; set; }
        public string EvaDate { get; set; }
        public Brush EvaBoxBrush { get; set; } = Brushes.Transparent;
        public string EvaPrefixText { get; set; }
        public List<EvaInfoViewModel> EvaInfo { get; set; } = [];
        public List<BuildInfoViewModel> BuildInfo { get; set; } = [];
        public List<ProdAnaInfoViewModel> ProdAnaInfo { get; set; } = [];
        public UnitInfo UnitInfo { get; set; }

        ~UnitInfoViewModel()
        {
            this.UnitInfo = null;
        }
        public UnitInfoViewModel(AnalyseToolExtension.DataObjects.UnitInfo unitInfo)
        {
            UnitId = unitInfo.UnitId;
            PrevId = unitInfo.PreUnitId;
            NextId = unitInfo.NextUnitId;
            IdType = unitInfo.UnitIdType;
            Material = unitInfo.Material;
            Description = unitInfo.Description;
            CarrierId = unitInfo.CarrierId;
            Order = unitInfo.Order;
            LockString = unitInfo.LockText;
            switch (unitInfo.WipMama)
            {
                case UnitStatus.Mama: { WipMamaBrush = new SolidColorBrush(Color.FromArgb(200,0,191,255)); break; }
                case UnitStatus.WipHistory:
                case UnitStatus.WipMama:
                { WipMamaBrush = new SolidColorBrush(Color.FromArgb(200,255,120,0)); break; }
            }
            WipMama = unitInfo.WipMama == UnitStatus.WipMama ? "WIP + MAMA" : unitInfo.WipMama.ToString("G").ToUpper();
            if (unitInfo.WipInfo?.Count > 0)
            {
                WipStation = $"{(unitInfo.WipInfo?[0]?.Equipment ?? string.Empty)} - {unitInfo.WipInfo?[0]?.StationDescription ?? string.Empty}";
                WipDate = $"{unitInfo.WipInfo?[0].Updated:yyyy-MM-dd HH:mm:ss}";
                switch (unitInfo.WipInfo?[0].Result)
                {
                    case UnitState.None: { WipPrefixText = string.Empty; WipBoxBrush = Brushes.Transparent; break;}
                    case UnitState.Pass: { WipPrefixText = "PASS"; WipBoxBrush = new SolidColorBrush(Color.FromArgb(200,0,238,0)); break;}
                    case UnitState.Fail: { WipPrefixText = "FAIL"; WipBoxBrush = new SolidColorBrush(Color.FromArgb(200,255,0,0)); break;}
                    case UnitState.Scrap: { WipPrefixText = "SCRAP"; WipBoxBrush = new SolidColorBrush(Color.FromArgb(200,255,120,0)); break;}
                    case UnitState.CheckIn: { WipPrefixText = "CHECKIN"; WipBoxBrush = new SolidColorBrush(Color.FromArgb(200,0,191,255)); break;}
                    case null: { WipPrefixText = string.Empty; WipBoxBrush = Brushes.Transparent; break;}
                }
            }

            if (unitInfo.EvaInfo?.Count > 0)
            {
                EvaStation =
                    $"{unitInfo.EvaInfo?[0].TestPlanName ?? string.Empty} - {unitInfo.EvaInfo?[0].Station ?? string.Empty}";
                EvaDate = $"{unitInfo.EvaInfo?[0].DateTime:yyyy-MM-dd HH:mm:ss}";
                switch (unitInfo.EvaInfo?[0].Result)
                {
                    case UnitState.None: { EvaPrefixText = string.Empty; EvaBoxBrush = Brushes.Transparent; break;}
                    case UnitState.Pass: { EvaPrefixText = "PASS"; EvaBoxBrush = new SolidColorBrush(Color.FromArgb(200,0,238,0)); break;}
                    case UnitState.Fail: { EvaPrefixText = "FAIL"; EvaBoxBrush = new SolidColorBrush(Color.FromArgb(200,255,0,0)); break;}
                    case UnitState.Scrap: { EvaPrefixText = "SCRAP"; EvaBoxBrush = new SolidColorBrush(Color.FromArgb(200,255,165,0)); break;}
                    case UnitState.CheckIn: { EvaPrefixText = "CHECKIN"; EvaBoxBrush = new SolidColorBrush(Color.FromArgb(200,0,191,255)); break;}
                    case null: { EvaPrefixText = string.Empty; EvaBoxBrush = Brushes.Transparent; break;}
                }
            }
            
            if (unitInfo.EvaInfo is { Count: > 0 })
            {
                foreach (var evaInfo in unitInfo.EvaInfo)
                {
                    EvaInfo.Add(new EvaInfoViewModel()
                    {
                        Date = evaInfo.DateTime?.ToString("yyyy-MM-dd HH:mm:ss"),
                        Type = evaInfo.UnitIdType,
                        Material = evaInfo.Material,
                        Result = evaInfo.Result.ToString("G").ToUpper(),
                        TestPlan = evaInfo.TestPlanName,
                        TestPlanVersion = evaInfo.TestPlanVersion,
                        Station = evaInfo.Station,
                        Key = evaInfo.KeyPrt,
                        Seq = evaInfo.SeqKey,
                        EvaInfo = evaInfo
                    });
                }
            }

            if (unitInfo.BuildInfos is { Count: > 0 })
            {
                foreach (var buildInfo in unitInfo.BuildInfos)
                {
                    var cre = buildInfo.Created?.ToString("yyyy-MM-dd HH:mm:ss");
                    var upd = buildInfo.Removed?.ToString("yyyy-MM-dd HH:mm:ss");
                    var pck = string.IsNullOrWhiteSpace(upd) ? buildInfo.IsPart ? PackIconKind.TrayArrowDown : PackIconKind.Import : buildInfo.IsPart ? PackIconKind.TrayArrowUp : PackIconKind.Export;
                    var dsc = buildInfo.Description;
                    var typ = buildInfo.UnitIdType;
                    var bdc = buildInfo.BuildCount;
                    var uid = buildInfo.UnitId;
                    var lst = string.IsNullOrWhiteSpace(upd) ? buildInfo.IsPart ? "Beépülő" : "Beépült" : buildInfo.IsPart ? "Kiépülő" : "Kiépült";
                    BuildInfo.Add(new BuildInfoViewModel()
                    {
                        UnitId = uid,
                        Created = cre,
                        Removed = upd,
                        PartIcon = pck,
                        PartName = dsc,
                        UnitIdType = typ,
                        LongStatusText = lst,
                        BuildCount = bdc.ToString(),
                        RowHeaderBrush = string.IsNullOrWhiteSpace(upd) ? new SolidColorBrush(Color.FromArgb(125,0,40,255)) : new SolidColorBrush(Color.FromArgb(125,255,40,0)),
                    });
                }
            }

            if (unitInfo.ProductAnalyzeInfo is { Count: > 0 })
            {
                foreach (var info in unitInfo.ProductAnalyzeInfo)
                {
                    var usr1 = info.Detection.UserFullName;
                    var dec = info.Detection.Date?.ToString("yyyy-MM-dd HH:mm:ss");
                    var stn = info.Detection.Station;
                    var err = info.Detection.Fault;
                    var st1 = info.Detection.Status;
                    var sat1 = info.Detection.AnalyzeState;
                    var usr2 = info.Causer.UserFullName;
                    var dec2 = info.Causer.Date?.ToString("yyyy-MM-dd HH:mm:ss");
                    var st2 = info.Causer.Status;
                    var err2 = info.Causer.Fault;
                    var sat2 = info.Causer.AnalyzeState;
                    var msg = info.Message;
                    ProdAnaInfo.Add(new ProdAnaInfoViewModel()
                    {
                        Detection = new ProdAnaRecordViewModel()
                        {
                            UserFullName = usr1,
                            Date = dec,
                            Station = stn,
                            Fault = err,
                            Status = st1,
                            AnalyzeState = sat1
                        },
                        Causer = new ProdAnaRecordViewModel()
                        {
                            UserFullName = usr2,
                            Date = dec2,
                            Station = st2,
                            Fault = err2,
                            Status = st2,
                            AnalyzeState = sat2
                        },
                        Message = msg,
                    });
                }
            }

            UnitInfo = unitInfo;
        }
    }

    public class EvaInfoViewModel
    {
        public string Date { get; set; }
        public string Type { get; set; }
        public string Material { get; set; }
        public string Result { get; set; }
        public string TestPlan { get; set; }
        public string TestPlanVersion { get; set; }
        public string Station { get; set; }
        public string Key { get; set; }
        public string Seq { get; set; }
        public EvaInfo EvaInfo { get; set; }
        public Brush RowHeaderBrush => EvaInfo.Result == UnitState.Fail ? new SolidColorBrush(Colors.Red) : new SolidColorBrush(Colors.Transparent);
    }

    public class BuildGroupViewModel
    {
        public string GroupName { get; set; }
        public List<BuildInfo> BuildInfos { get; set; }
    }
    public class BuildInfoViewModel
    {
        public PackIconKind PartIcon { get; set; }
        public string LongStatusText { get; set; }
        public string UnitId { get; set; }
        public string UnitIdType { get; set; }
        public string Created { get; set; }
        public string Removed { get; set; }
        public string Order { get; set; }
        public string PartNumber { get; set; }
        public string PartName { get; set; }
        public string PartType { get; set; }
        public string BuildCount { get; set; }
        public Color RowColor { get; set; } = Colors.Transparent;
        public Brush RowBrush => new SolidColorBrush(RowColor);
        public Color FontColor { get; set; }
        public Brush FontBrush => new SolidColorBrush(FontColor);
        public Brush RowHeaderBrush { get; set; }
    }

    public class ProdAnaInfoViewModel
    {
        public string Plant { get; set; }
        public string Building { get; set; }
        public string FocusFactory  { get; set; }
        public string ProductFamily { get; set; }
        public string UnitId { get; set; }
        public string UnitIdType { get; set; }
        public string Material { get; set; }
        public string Subset { get; set; }
        public string Location { get; set; }
        public string Order { get; set; }
        public string ProductionVersion { get; set; }
        public string Customer  { get; set; }
        public string ProductType { get; set; }
        public string SubGroup { get; set; }
        public string Message { get; set; }
        public string Protocol { get; set; }
        public string ProtocolType { get; set; }
        public string CostCollection  { get; set; }
        public string Price  { get; set; }
        public string ComputerName { get; set; }
        public string MesStation { get; set; }
        public string MesDate { get; set; }
        public ProdAnaRecordViewModel Detection { get; set; }
        public ProdAnaRecordViewModel Causer { get; set; }
    }

    public class ProdAnaRecordViewModel
    {
        public string Building { get; set; }
        public string Production { get; set; }
        public string Line { get; set; }
        public string Station { get; set; }
        public string OperationNumber { get; set; }
        public string WorkCenterGroup  { get; set; }
        public string Fault  { get; set; }
        public string Params { get; set; }
        public string Status { get; set; }
        public PackIconKind StatusIcon { get; set; }
        public string AnalyzeState { get; set; }
        public string Date {get;set;}
        public string Week { get; set; }
        public string Shift { get; set; }
        public string User { get; set; }
        public string UserFullName { get; set; }
    }
}