using System;
using System.Collections.Generic;
using System.Linq;
using System.Web.UI.HtmlControls;
using System.Windows;
using System.Windows.Media;
using AndonMonitor.StaticWorkers;
using MaterialDesignThemes.Wpf;

namespace AndonMonitor.ViewModels
{
    public class FluentViewModel
    {
        public string Line { get; set; }
        public string Building { get; set; }
        public string Shift { get; set; }
        public string Product { get; set; }
        public string ShiftPlan { get; set; }
        public string ShiftExpect { get; set; }
        public string ShiftActual { get; set; }
        public string ShiftNok { get; set; }
        public string ShiftStatus { get; set; }
        public Brush ShiftBrush { get; set; }
        public Color ShiftBrushToColor => ShiftBrush is SolidColorBrush sbr ? sbr.Color : Colors.Transparent; 
        public string HourPlan { get; set; }
        public string HourExpect { get; set; }
        public string HourActual { get; set; }
        public string HourNok { get; set; }
        public string HourStatus { get; set; }
        public Brush HourBrush { get; set; }
        public Color HourBrushToColor => HourBrush is SolidColorBrush sbr ? sbr.Color : Colors.Transparent; 
        public Brush HourForeground { get; set; }
        
        public List<FluentViewModelHour> Targets { get; set; }
        public List<FluentViewModelStation> Stations { get; set; }
        public string CurrentTime { get; set; } = $"TIME {DateTime.Now:HH:mm:ss}";

        public string LeftTime { get; set; } = $"LEFT {(DateTime.Now - StaticValues.ShiftEndTime).Duration():hh\\:mm\\:ss}";
        public string ShiftPercentage { get; set; }
        public double ShiftPercent { get; set; } 
        public PackIconKind ShiftStatusIcon { get; set; }
        public Visibility ShiftStatusIconVisibility => ShiftStatusIcon == PackIconKind.MenuUp || ShiftStatusIcon == PackIconKind.MenuDown ? Visibility.Visible : Visibility.Collapsed;
        public PackIconKind HourStatusIcon { get; set; }
        public Visibility HourStatusIconVisibility => HourStatusIcon == PackIconKind.MenuUp || HourStatusIcon == PackIconKind.MenuDown ? Visibility.Visible : Visibility.Collapsed;

        public FluentViewModel(AnalyseToolExtension.DataObjects.Andon andon)
        {
            var shiftPlan = 0;
            var shiftExpect = 0;
            var shiftActual = 0;
            var shiftNok = 0;
            var shiftStatus = 0;
            
            var hourPlan = 0;
            var hourExpect = 0;
            var hourActual = 0;
            var hourNok = 0;
            var hourStatus = 0;
            
            HourForeground = new SolidColorBrush(Colors.White);
            
            #region LineHeaders
            
            Line = andon.Definition.Name;
            Building = andon.Definition.Building;
            var mat = andon.Definition.ProductIn;
            if (andon.Definition.ProductIn != andon.Definition.ProductOut) mat = string.Join(", ", andon.Definition.ProductIn, andon.Definition.ProductOut);
            Product = mat;
            
            #endregion
            #region HourTargets

            Targets = andon.Targets.Select(tg =>
            {
                var hrm = new FluentViewModelHour();
                var hc = tg.Hour?.Hour ?? 0;
                var hp = tg.Hour.HasValue ? tg.Hour.Value.Hour + 1 : 0;
                hrm.HourString = $"{hc:00}-{hp:00}";
                hrm.YieldString = tg.YieldQty.ToString();
                var maxDiv = Math.Ceiling((double)tg.Target * 0.1);
                if (tg.YieldQty >= tg.Target - maxDiv)
                {
                    hrm.Color = StaticValues.GetBr(StaticBr.Yellow);
                    hrm.Foreground = StaticValues.GetBr(StaticBr.Yellow);
                }
                else { hrm.Color = StaticValues.GetBr(StaticBr.Red); hrm.Foreground = StaticValues.GetBr(StaticBr.Red); }
                if (tg.YieldQty >= tg.Target) { hrm.Color = StaticValues.GetBr(StaticBr.Green); hrm.Foreground = StaticValues.GetBr(StaticBr.Green);}
                if (tg.Target == 0){ hrm.Color = StaticValues.GetBr(StaticBr.Blue); StaticValues.GetBr(StaticBr.Blue);}
                if (DateTime.Now < tg.Hour) { hrm.Color = StaticValues.GetBr(StaticBr.Gray); hrm.Foreground = StaticValues.GetBr(StaticBr.Gray);}

                if (DateTime.Now.Year == tg.Hour?.Year && DateTime.Now.Month == tg.Hour?.Month &&
                    DateTime.Now.Day == tg.Hour?.Day && DateTime.Now.Hour == tg.Hour?.Hour)
                {
                    hrm.IsCurrent = true;  
                    //hrm.Color = new SolidColorBrush(Colors.Transparent);
                    hrm.Foreground = new SolidColorBrush(Colors.Transparent);
                    hourActual = tg.YieldQty;
                    hourPlan = tg.Target;
                }

                hrm.Target = tg.Target.ToString();

                var eqs = tg.YieldQty-tg.Target;
                if (eqs > 0) { hrm.Status = $"+{eqs}"; hrm.StatusIcon = PackIconKind.MenuUp; }
                if (eqs < 0) { hrm.Status = $"{eqs}"; hrm.StatusIcon = PackIconKind.MenuDown; }
                if (eqs == 0) { hrm.Status = "0"; hrm.StatusIcon = PackIconKind.MenuLeft; }
                
                return hrm;
            }).ToList();

            #endregion
            
            #region HourValues

            const int whs = 3600;
            var dtn = DateTime.Now;
            var ehs = (double)(dtn - new DateTime(dtn.Year, dtn.Month, dtn.Day,dtn.Hour,0,0)).TotalSeconds;
            if (ehs < 0 || ehs > whs)
            {
                if (ehs < 0) { hourExpect = 0; }
                if (ehs > whs) { hourExpect = hourPlan; }
            }
            else
            {
                hourExpect = int.Parse($"{Math.Floor(((double)hourPlan / whs)*ehs):0000}");
            }
            hourStatus = hourActual - hourExpect;
            
            HourActual = hourActual.ToString();
            HourPlan = hourPlan.ToString();
            HourExpect = hourExpect.ToString();
            HourNok = hourNok.ToString();
            HourStatus = hourStatus > 0 ? $"+{hourStatus}" : $"{hourStatus}";
            

            var hourDiv = Math.Ceiling((double)hourExpect * 0.1);
            if (hourActual >= hourExpect - hourDiv)
            {
                HourBrush = StaticValues.GetBr(hourStatus >= 0 ? StaticBr.Green : StaticBr.Yellow);
                HourForeground = new SolidColorBrush(Colors.Black);
            }
            else { HourBrush = StaticValues.GetBr(StaticBr.Red); }
            if (hourPlan == 0){HourBrush = StaticValues.GetBr(StaticBr.Blue); HourForeground = new SolidColorBrush(Colors.White);}
            
            var cht = new FluentViewModelHour
            {
                Color = Brushes.Transparent,
                Foreground = HourBrush,
                Target = HourPlan,
                Status = HourStatus,
                YieldString = HourActual,
                IsCurrent = true
            };
            var eqs2 = hourActual-hourExpect;
            if (eqs2 > 0) { cht.StatusIcon = PackIconKind.MenuUp; HourStatusIcon = PackIconKind.MenuUp; }
            if (eqs2 < 0) { cht.StatusIcon = PackIconKind.MenuDown; HourStatusIcon = PackIconKind.MenuDown; }
            if (eqs2 == 0) { cht.StatusIcon = PackIconKind.MenuLeft; HourStatusIcon = PackIconKind.MenuLeft; }
            Targets.Find(tg => tg.IsCurrent).CopyModel(cht);
            
            #endregion
            #region ShiftValues
            
            shiftPlan = andon.Targets.Sum(tg=> tg.Target);
            shiftActual = andon.Targets.Sum(tg => tg.YieldQty);
            shiftExpect = GetShiftCurrentTargets() + hourExpect;
            shiftStatus = shiftActual - shiftExpect;
            
            ShiftPlan = shiftPlan.ToString();
            ShiftExpect = shiftExpect.ToString();
            ShiftActual = shiftActual.ToString();
            ShiftNok = shiftNok.ToString();
            ShiftStatus = shiftStatus > 0 ? $"+{shiftStatus}" : $"{shiftStatus}";

            var shiftDiv = Math.Ceiling((double)shiftExpect * 0.1);
            ShiftBrush = shiftActual >= shiftExpect - shiftDiv ? shiftStatus >= 0 ? StaticValues.GetBr(StaticBr.Green) : StaticValues.GetBr(StaticBr.Yellow) : StaticValues.GetBr(StaticBr.Red) ;

            if (shiftExpect != 0)
            {
                var spt = (shiftActual * 100) / shiftExpect;
                ShiftPercentage = $"{spt}%";
                ShiftPercent = spt;
            }
            
            var eqs3 = shiftActual-shiftExpect;
            if (eqs3 > 0) { ShiftStatusIcon = PackIconKind.MenuUp; }
            if (eqs3 < 0) { ShiftStatusIcon = PackIconKind.MenuDown; }
            if (eqs3 == 0) { ShiftStatusIcon = PackIconKind.MenuLeft; }

            #endregion

            #region Stations

            if (andon.Stations != null && andon.Stations.Count > 0)
            {
                var stf = new List<FluentViewModelStation>();
                foreach (var station in andon.Stations)
                {
                    var rts = string.Empty;
                    var sbr = StaticValues.GetBr(StaticBr.Gray);
                    if (station.LastUsed != null)
                    {
                        var lu = station.LastUsed.Value;
                        var tip = lu - DateTime.Now;
                        
                        if (Math.Abs(tip.TotalSeconds) > station.FirstEscalation)
                        {
                            rts = $"{tip.Seconds}sec";
                            if (tip.Minutes <= -1) { rts = $"{tip.Minutes}m {tip.Seconds}sec"; }
                            if (tip.Hours <= -1) { rts = $"{tip.Hours}h {tip.Minutes}m"; }
                            if (tip.Days <= -1) { rts = $"{tip.Days}d {tip.Hours}h"; }
                            if (tip.Days <= -365) { rts = $"{tip.Days}d"; }
                        }
                        sbr = Math.Abs(tip.TotalSeconds) > station.FirstEscalation ? 
                            StaticValues.GetBr(StaticBr.Red) : StaticValues.GetBr(StaticBr.Green);
                    }
                    
                    
                    
                    if (stf.Find(s => s.Name.Equals(station.Name) && (s.Equipment.Equals(station.Equipment)|| s.Operator.Equals(station.Operator))) is FluentViewModelStation st)
                    {
                        st.FirstPass = (st.FirstPassToInt + station.FirstPass).ToString();
                        st.FirstFail = (st.FirstFailToInt + station.FirstFail).ToString();
                        st.Fpy = $"{(((double)st.FirstPassToInt / (st.FirstPassToInt + st.FirstFailToInt)) * 100):0.#}%";
                        if(st.IsLocked == false){st.IsLocked = station.IsLocked;}
                    }
                    else
                    {
                        var stl = new FluentViewModelStation
                        {
                            Name = station.Name,
                            Operator = station.Operator,
                            Equipment = station.Equipment,
                            IsLast = station.IsLast,
                            IsLocked = station.IsLocked,
                            StatusColor = sbr,
                            TimeLeft = rts,
                            Fpy = $"{station.TotalFpyPercentage:0.#}%",
                            FirstPass = station.FirstPass.ToString(),
                            FirstFail = station.FirstFail.ToString(),
                        };
                        stf.Add(stl);
                    }
                    
                }

                //stf[0].IsLocked = true; //DEBUG JIDOKA
                Stations = stf;
            }

            #endregion
        }

        private int GetShiftCurrentTargets()
        {
            var rtn = 0;
            foreach (var target in Targets)
            {
                if (!target.IsCurrent)
                {
                    rtn += int.TryParse(target.Target, out var ts) ? ts : 0;
                }
                else
                {
                    break;
                }
            }
            return rtn;
        }
    }
    
    public class FluentViewModelHour
    {
        public string HourString { get; set; }
        public string YieldString { get; set; }
        public Brush Color { get; set; }
        public Color ColorToColor => Color is SolidColorBrush br ? br.Color : Colors.Transparent;
        public Brush Foreground { get; set; } = new SolidColorBrush(Colors.White);
        public Color ForegroundToColor => Foreground is SolidColorBrush br ? br.Color : Colors.Transparent;
        public bool IsCurrent { get; set; }
        public string Target { get; set; }
        public string Status { get; set; }
        public PackIconKind StatusIcon { get; set; }
        public Visibility StatusIconVisibility => StatusIcon == PackIconKind.MenuUp || StatusIcon == PackIconKind.MenuDown ? Visibility.Visible : Visibility.Collapsed;

        public void CopyModel(FluentViewModelHour model)
        {
            //this.HourString = model.HourString;
            this.YieldString = model.YieldString;
            this.Color = model.Color;
            this.Foreground = model.Foreground;
            this.IsCurrent = model.IsCurrent;
            this.Target = model.Target;
            this.Status = model.Status;
            this.StatusIcon = model.StatusIcon;
        }
        
    }

    public class FluentViewModelStation
    {
        public string Name { get; set; }
        public string Operator { get; set; }
        public string Equipment { get; set; }
        public string TimeLeft { get; set; }
        public bool? IsLocked { get; set; }
        public bool? IsLast { get; set; }
        public Brush StatusColor { get; set; }
        public Color StatusColorToColor => StatusColor is SolidColorBrush br ? br.Color : Colors.Transparent;
        public string Fpy { get; set; } = "0%";
        public string FirstFail { get; set; } = "0";
        public int FirstFailToInt => int.TryParse(FirstFail, out var i) ? i : 0;
        public string FirstPass { get; set; } = "0";
        public int FirstPassToInt => int.TryParse(FirstPass, out var i) ? i : 0;
    }
}